/*
 * ProjectileDamageObject.h
 *
 * Created 9/12/2009 By Johnny Huynh
 *
 * Version 00.00.01 9/12/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef PROJECTILE_DAMAGE_OBJECT_H
 #define PROJECTILE_DAMAGE_OBJECT_H
 
 template <typename T> class ProjectileDamageObject;
 
 #include "CollisionHandlerEventNode.h"
 #include "DamageObject.h"
 
 #include "global.h"
 
 /**
  * Class specification for ProjectileDamageObject
  * A ProjectileDamageObject is a projectile that deals damage
  */
 template <typename T>
 class ProjectileDamageObject : virtual public DamageObject<T>
 {
 // Data Members
 private:
 
 // Local Functions
 public:
    ProjectileDamageObject( const std::string& name, const T& damage = ZERO ); 
    ProjectileDamageObject( const std::string& name, const T& damage, const EffectInfoCollection<T>& effect_info_collection ); 
    ProjectileDamageObject( const ProjectileDamageObject<T>& proj_dmg_obj );
    virtual ~ProjectileDamageObject();
    inline ProjectileDamageObject<T>& operator=( const ProjectileDamageObject<T>& proj_dmg_obj );
    virtual inline DamageObject<T>* get_damage_object( const std::string& key );
    virtual inline void handle_from_collision( Object<T>& into, const CollisionEntry& c_entry );
    virtual inline void handle_into_collision( Object<T>& from, const CollisionEntry& c_entry );
    virtual inline bool is_effected_by_gravity() const;
    
    // overloaded functions (NodePath)
    /*static void init_type() {
                                std::string template_type( typeid( T ).name() );
                                register_type(_type_handle, "ProjectileDamageObject<" + template_type + ">" );
                            }*/
 
 // Private Functions
 private:
    
 // Public Static Functions
 public:
    
 };
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 ProjectileDamageObject<T>::ProjectileDamageObject( const std::string& name, const T& damage )
                           : DamageObject<T>( name, damage )
 {
    
 }
 
 /**
  * Alternative Constructor
  */
 template <typename T>
 ProjectileDamageObject<T>::ProjectileDamageObject( const std::string& name, const T& damage, const EffectInfoCollection<T>& effect_info_collection )
                           : DamageObject<T>( name, damage, effect_info_collection )
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 ProjectileDamageObject<T>::ProjectileDamageObject( const ProjectileDamageObject<T>& proj_dmg_obj )
                           : DamageObject<T>( proj_dmg_obj )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 ProjectileDamageObject<T>::~ProjectileDamageObject()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified ProjectileDamageObject to this ProjectileDamageObject.
  *
  * @param (const ProjectileDamageObject<T>& proj_dmg_obj )
  * @return ProjectileDamageObject<T>&
  */
 template <typename T>
 inline ProjectileDamageObject<T>& ProjectileDamageObject<T>::operator=( const ProjectileDamageObject<T>&  proj_dmg_obj )
 {
    DamageObject<T>::operator=( static_cast< DamageObject<T> >( proj_dmg_obj ) );
    
    return *this;
 }
 
 /**
  * get_damage_object() returns a pointer to this DamageObject regardless of the specified key.
  *
  * @param (const std::string&) key
  * @return DamageObject<T>*
  */
 template <typename T>
 inline DamageObject<T>* ProjectileDamageObject<T>::get_damage_object( const std::string& key )
 {
    return this;
 }
 
 /**
  * handle_from_collision() handles the collision assuming this Object has
  * collided into another Object (additional information can be found 
  * inside of the CollisionEntry). That is, this Object is assumed to be
  * the "from" Object.
  *
  * @param (Object<T>&) into - the "into" Object
  * @param (const CollisionEntry&) c_entry
  */
 template <typename T>
 inline void ProjectileDamageObject<T>::handle_from_collision( Object<T>& into, const CollisionEntry& c_entry )
 {
    DamageObject<T>::handle_from_collision( into, c_entry );
    
    // clear tasks (to self destruct)
    Object<T>::clear_tasks();
 }
 
 /**
  * handle_into_collision() handles the collision assuming another Object
  * has collided into this Object (additional information can be found 
  * inside of the CollisionEntry). That is, this Object is assumed to be
  * the "into" Object.
  *
  * @param (Object<T>&) from - the "from" Object
  * @param (const CollisionEntry&) c_entry
  */
 template <typename T>
 inline void ProjectileDamageObject<T>::handle_into_collision( Object<T>& from, const CollisionEntry& c_entry )
 {
    DamageObject<T>::handle_into_collision( from, c_entry );
    
    // clear tasks (to self destruct)
    Object<T>::clear_tasks();
 }
 
 /**
  * is_effected_by_gravity() returns true if this Object is effected
  * by gravity; otherwise, false is returned.
  *
  * @return bool
  */
 template <typename T>
 inline bool ProjectileDamageObject<T>::is_effected_by_gravity() const
 {
    return false;
 }
 
 #endif // PROJECTILE_DAMAGE_OBJECT_H